// Excursion Booking Website - Version 3 (HTML/CSS/JS) - Bright Tourist Design

const themes = {
  historical: {
    name: 'Историческая',
    color: '#2E86AB',
    gradient: 'linear-gradient(135deg, #2E86AB 0%, #1E5F7A 100%)',
    description: 'Погрузитесь в многовековую историю города',
    duration: '3 часа',
    price: 'от 1500₽',
    features: ['Эксклюзивные маршруты', 'Исторические факты', 'Архитектурные детали'],
    guide: 'Анна Петрова'
  },
  gastronomic: {
    name: 'Гастрономическая',
    color: '#A23B72',
    gradient: 'linear-gradient(135deg, #A23B72 0%, #7A2B55 100%)',
    description: 'Откройте вкусы и традиции местной кухни',
    duration: '2.5 часа',
    price: 'от 2000₽',
    features: ['Дегустации', 'Исторические рецепты', 'Встречи с шеф-поварами'],
    guide: 'Михаил Соколов'
  },
  nature: {
    name: 'Природная',
    color: '#06A77D',
    gradient: 'linear-gradient(135deg, #06A77D 0%, #048A68 100%)',
    description: 'Насладитесь красотой природы в черте города',
    duration: '4 часа',
    price: 'от 1800₽',
    features: ['Экологические тропы', 'Наблюдение за природой', 'Фотосессии'],
    guide: 'Елена Волкова'
  },
  architectural: {
    name: 'Архитектурная',
    color: '#F18F01',
    gradient: 'linear-gradient(135deg, #F18F01 0%, #D67E00 100%)',
    description: 'Изучите уникальную архитектуру разных эпох',
    duration: '3.5 часа',
    price: 'от 1700₽',
    features: ['Архитектурные стили', 'Секреты строительства', 'Современные проекты'],
    guide: 'Дмитрий Новиков'
  }
}

let currentSlide = 0
let slides = []
let dots = []
let slideInterval
let selectedTour = null
let selectedTheme = ''
let selectedDate = ''

function initPage() {
  const currentPage = window.location.pathname.includes('book.html') ? 'book' : 'index'
  
  if (currentPage === 'book') {
    initBookPage()
  } else {
    initIndexPage()
  }
  
  initSlider()
}

function initIndexPage() {
  loadTours()
  updateStats()
  
  setInterval(() => {
    loadTours()
    updateStats()
  }, 500)
}

function loadTours() {
  const saved = localStorage.getItem('excursionsTours')
  let tours = []
  
  if (saved) {
    try {
      tours = JSON.parse(saved)
    } catch (e) {
      tours = []
    }
  } else {
    tours = [
      {
        id: 1,
        route: 'Тайны Красной площади',
        guide: 'Анна Петрова',
        date: '2025-04-15',
        rating: 5,
        duration: '3 часа',
        theme: 'Историческая',
        highlights: ['Кремль', 'Собор Василия Блаженного', 'ГУМ']
      },
      {
        id: 2,
        route: 'Гастрономические секреты Замоскворечья',
        guide: 'Михаил Соколов',
        date: '2025-04-12',
        rating: 4,
        duration: '2.5 часа',
        theme: 'Гастрономическая',
        highlights: ['Традиционные блюда', 'Исторические кафе', 'Дегустации']
      },
      {
        id: 3,
        route: 'Природные сокровища Коломенского',
        guide: 'Елена Волкова',
        date: '2025-04-10',
        rating: 5,
        duration: '4 часа',
        theme: 'Природная',
        highlights: ['Древние дубы', 'Яблоневые сады', 'Речные пейзажи']
      }
    ]
    localStorage.setItem('excursionsTours', JSON.stringify(tours))
  }
  
  renderTours(tours)
}

function renderTours(tours) {
  const container = document.getElementById('tours-container')
  if (!container) return
  
  if (tours.length === 0) {
    container.innerHTML = `
      <div class="empty-immersive">
        <div class="empty-compass">
          <img src="icons/compass.svg" alt="Compass" width="64" height="64">
          <div class="compass-ring"></div>
        </div>
        <h3 class="empty-title-immersive">Начните свое путешествие</h3>
        <p class="empty-text-immersive">Запишитесь на первую экскурсию и откройте для себя новые горизонты</p>
      </div>
    `
    return
  }
  
  container.innerHTML = tours.map(tour => {
    const isExpanded = selectedTour === tour.id
    const date = new Date(tour.date)
    const formattedDate = date.toLocaleDateString('ru-RU', { day: 'numeric', month: 'long' })
    
    return `
      <div class="tour-map-card ${isExpanded ? 'expanded' : ''}" data-id="${tour.id}">
        <div class="tour-map-marker">
          <img src="icons/map-pin.svg" alt="Location" width="20" height="20">
          <div class="marker-pulse"></div>
        </div>
        <div class="tour-map-content">
          <div class="tour-map-header">
            <div class="tour-theme-badge" style="background: var(--gradient-1);">
              ${tour.theme}
            </div>
            <div class="tour-rating-compact">
              ${Array.from({ length: 5 }).map((_, i) => 
                `<img src="icons/star.svg" alt="Star" width="16" height="16" style="filter: ${i < tour.rating ? 'none' : 'grayscale(100%) opacity(0.3)'}">`
              ).join('')}
            </div>
          </div>
          <h3 class="tour-route-map">${tour.route}</h3>
          <div class="tour-info-grid">
            <div class="tour-info-item">
              <img src="icons/user.svg" alt="User" width="18" height="18">
              <span>${tour.guide}</span>
            </div>
            <div class="tour-info-item">
              <img src="icons/calendar.svg" alt="Calendar" width="18" height="18">
              <span>${formattedDate}</span>
            </div>
            <div class="tour-info-item">
              <img src="icons/clock.svg" alt="Clock" width="18" height="18">
              <span>${tour.duration}</span>
            </div>
          </div>
          ${isExpanded && tour.highlights ? `
            <div class="tour-highlights">
              <div class="highlights-title">
                <img src="icons/route.svg" alt="Route" width="18" height="18">
                <span>Ключевые точки</span>
              </div>
              <div class="highlights-list">
                ${tour.highlights.map(h => `<div class="highlight-item">${h}</div>`).join('')}
              </div>
            </div>
          ` : ''}
        </div>
      </div>
    `
  }).join('')
  
  // Add click handlers
  container.querySelectorAll('.tour-map-card').forEach(card => {
    card.addEventListener('click', () => {
      const id = parseInt(card.dataset.id)
      selectedTour = selectedTour === id ? null : id
      loadTours()
    })
  })
}

function updateStats() {
  const saved = localStorage.getItem('excursionsTours')
  let tours = []
  
  if (saved) {
    try {
      tours = JSON.parse(saved)
    } catch (e) {
      tours = []
    }
  }
  
  const totalTours = tours.length
  const averageRating = tours.length > 0
    ? (tours.reduce((sum, t) => sum + t.rating, 0) / tours.length).toFixed(1)
    : 0
  const upcomingTours = tours.filter(t => new Date(t.date) >= new Date()).length
  const completedTours = tours.filter(t => new Date(t.date) < new Date()).length
  
  const totalEl = document.getElementById('total-tours')
  const avgEl = document.getElementById('avg-rating')
  const upcomingEl = document.getElementById('upcoming-tours')
  const completedEl = document.getElementById('completed-tours')
  
  if (totalEl) totalEl.textContent = totalTours
  if (avgEl) avgEl.textContent = averageRating
  if (upcomingEl) upcomingEl.textContent = upcomingTours
  if (completedEl) completedEl.textContent = completedTours
}

function initSlider() {
  slides = document.querySelectorAll('.slide')
  dots = document.querySelectorAll('.dot')
  
  if (slides.length === 0) return
  
  const prevBtn = document.getElementById('slider-prev')
  const nextBtn = document.getElementById('slider-next')
  
  if (prevBtn) {
    prevBtn.addEventListener('click', () => {
      currentSlide = (currentSlide - 1 + slides.length) % slides.length
      updateSlider()
    })
  }
  
  if (nextBtn) {
    nextBtn.addEventListener('click', () => {
      currentSlide = (currentSlide + 1) % slides.length
      updateSlider()
    })
  }
  
  dots.forEach((dot, index) => {
    dot.addEventListener('click', () => {
      currentSlide = index
      updateSlider()
    })
  })
  
  updateSlider()
  
  slideInterval = setInterval(() => {
    currentSlide = (currentSlide + 1) % slides.length
    updateSlider()
  }, 3000)
}

function updateSlider() {
  slides.forEach((slide, index) => {
    if (index === currentSlide) {
      slide.classList.add('active')
    } else {
      slide.classList.remove('active')
    }
  })
  
  dots.forEach((dot, index) => {
    if (index === currentSlide) {
      dot.classList.add('active')
    } else {
      dot.classList.remove('active')
    }
  })
}

function initBookPage() {
  renderThemes()
  
  const dateInput = document.getElementById('date-input')
  if (dateInput) {
    const today = new Date()
    const maxDate = new Date(today.getTime() + 90 * 24 * 60 * 60 * 1000)
    dateInput.min = today.toISOString().split('T')[0]
    dateInput.max = maxDate.toISOString().split('T')[0]
    
    dateInput.addEventListener('change', (e) => {
      selectedDate = e.target.value
      updateFormProgress()
      toggleDateSection(true)
    })
  }
  
  const form = document.getElementById('book-form')
  if (form) {
    form.addEventListener('submit', handleSubmit)
  }
  
  updateFormProgress()
}

function renderThemes() {
  const container = document.getElementById('themes-container')
  if (!container) return
  
  container.innerHTML = Object.entries(themes).map(([id, theme]) => {
    const isSelected = selectedTheme === id
    return `
      <button type="button" class="theme-wheel-card ${isSelected ? 'selected' : ''}" 
              data-theme="${id}" 
              style="${isSelected ? `background: ${theme.gradient}; border-color: ${theme.color}` : ''}">
        <div class="theme-wheel-icon" style="border-color: ${theme.color}; background: ${isSelected ? 'rgba(255, 255, 255, 0.2)' : 'rgba(46, 134, 171, 0.1)'}">
          <img src="icons/${id === 'historical' ? 'building2' : id === 'gastronomic' ? 'utensils-crossed' : id === 'nature' ? 'tree-pine' : 'camera'}.svg" 
               alt="${theme.name}" width="56" height="56" style="filter: ${isSelected ? 'brightness(0) invert(1)' : 'none'}">
          ${isSelected ? '<div class="icon-glow"></div>' : ''}
        </div>
        <div class="theme-wheel-content">
          <h3 class="theme-wheel-name" style="color: ${isSelected ? 'white' : 'var(--text)'}">
            ${theme.name}
          </h3>
          <p class="theme-wheel-description" style="color: ${isSelected ? 'rgba(255, 255, 255, 0.9)' : 'var(--text-light)'}">
            ${theme.description}
          </p>
          <div class="theme-wheel-features">
            ${theme.features.map(f => `
              <div class="feature-tag" style="background: ${isSelected ? 'rgba(255, 255, 255, 0.2)' : 'rgba(46, 134, 171, 0.08)'}; color: ${isSelected ? 'white' : 'var(--primary)'}">
                ${f}
              </div>
            `).join('')}
          </div>
          <div class="theme-wheel-meta">
            <div class="meta-item">
              <img src="icons/clock.svg" alt="Clock" width="16" height="16">
              <span>${theme.duration}</span>
            </div>
            <div class="meta-item">
              <span>${theme.price}</span>
            </div>
          </div>
        </div>
        ${isSelected ? `
          <div class="theme-check-wheel">
            <img src="icons/check-circle.svg" alt="Check" width="36" height="36">
          </div>
        ` : ''}
      </button>
    `
  }).join('')
  
  container.querySelectorAll('.theme-wheel-card').forEach(card => {
    card.addEventListener('click', () => {
      selectedTheme = card.dataset.theme
      renderThemes()
      updateFormProgress()
      toggleDateSection(true)
      updateProgressSteps(2)
    })
  })
}

function toggleDateSection(show) {
  const section = document.getElementById('date-section')
  if (section) {
    if (show && selectedTheme) {
      section.style.display = 'block'
      setTimeout(() => {
        section.style.opacity = '1'
        section.style.height = 'auto'
      }, 10)
    } else {
      section.style.opacity = '0'
      section.style.height = '0'
      setTimeout(() => {
        section.style.display = 'none'
      }, 300)
    }
  }
}

function updateFormProgress() {
  let progress = 0
  if (selectedTheme) progress += 50
  if (selectedDate) progress += 50
  
  const progressBar = document.getElementById('progress-fill')
  if (progressBar) {
    progressBar.style.width = `${progress}%`
  }
  
  const progressText = document.getElementById('progress-text')
  if (progressText) {
    progressText.textContent = `${progress}%`
  }
  
  const submitBtn = document.getElementById('submit-btn')
  if (submitBtn) {
    submitBtn.disabled = !selectedTheme || !selectedDate
  }
}

function updateProgressSteps(step) {
  document.querySelectorAll('.progress-step').forEach((el, index) => {
    if (index < step) {
      el.classList.add('active')
    } else {
      el.classList.remove('active')
    }
  })
}

function handleSubmit(e) {
  e.preventDefault()
  
  if (!selectedTheme || !selectedDate) return
  
  const saved = localStorage.getItem('excursionsTours')
  const tours = saved ? JSON.parse(saved) : []
  
  const themeData = themes[selectedTheme]
  
  const newTour = {
    id: Date.now(),
    route: themeData.name,
    guide: themeData.guide,
    date: selectedDate,
    rating: 0,
    duration: themeData.duration,
    theme: themeData.name,
    highlights: themeData.features
  }
  
  tours.push(newTour)
  localStorage.setItem('excursionsTours', JSON.stringify(tours))
  
  const successMsg = document.getElementById('success-message')
  if (successMsg) {
    successMsg.style.display = 'flex'
  }
  
  setTimeout(() => {
    window.location.href = 'index.html'
  }, 2000)
}

// Initialize on page load
document.addEventListener('DOMContentLoaded', initPage)

